/*:
 * @target MZ
 * @plugindesc v1.3 高速モード＋自動送り。重要シーン「保護」をオプションで切替可（デフォルトON）
 * @author HS
 *
 * @param OptionName
 * @text オプション表示名（高速）
 * @type string
 * @default 高速モード
 *
 * @param ImportantSwitchId
 * @text 重要シーン切替スイッチID
 * @type switch
 * @default 0
 * @desc このスイッチON中は（保護が有効のとき）高速/自動送りを無効化
 *
 * @param ProtectOptionName
 * @text オプション表示名（重要シーンを保護）
 * @type string
 * @default 重要シーンを保護
 *
 * @param ProtectDefault
 * @text 初期状態（重要シーンを保護）
 * @type boolean
 * @on ON
 * @off OFF
 * @default true
 *
 * @param ForceMessageFast
 * @text メッセージ常時早送り
 * @type boolean
 * @on 有効
 * @off 無効
 * @default true
 *
 * @param MessageWaitDivisor
 * @text メッセージ待機 短縮率（割り数）
 * @type number
 * @min 1
 * @default 2
 *
 * @param InterpreterWaitDivisor
 * @text イベント待機 短縮率（割り数）
 * @type number
 * @min 1
 * @default 2
 *
 * @param AutoOptionName
 * @text オプション表示名（自動送り）
 * @type string
 * @default 自動送り
 *
 * @param AutoEnabledByDefault
 * @text 初期状態（自動送り）
 * @type boolean
 * @on ON
 * @off OFF
 * @default false
 *
 * @param AutoBaseWait
 * @text 自動送り 基本待機(フレーム)
 * @type number
 * @min 0
 * @default 30
 *
 * @param AutoPerCharWait
 * @text 自動送り 文字あたり待機(フレーム)
 * @type number
 * @min 0
 * @default 1
 *
 * @param AutoMinWait
 * @text 自動送り 最小待機(フレーム)
 * @type number
 * @min 0
 * @default 20
 *
 * @param AutoMaxWait
 * @text 自動送り 最大待機(フレーム)
 * @type number
 * @min 0
 * @default 240
 *
 * @param AutoHotkeyA
 * @text Aキーで自動送り切替
 * @type boolean
 * @on 有効
 * @off 無効
 * @default true
 *
 * @help
 * ■概要
 * ・オプションに「高速モード」「自動送り」「重要シーンを保護」を追加。
 * ・「重要シーンを保護」ON（既定）：ImportantSwitchId のスイッチがONの間は高速/自動送りを停止。
 * ・OFF：スイッチがONでも高速/自動を停止しません（＝重要シーンも流れます）。ご利用は自己責任で。
 *
 * ■導入
 * 1) js/pluginsに入れてON
 * 2) ImportantSwitchId を設定（例: 15）。演出前ON→演出後OFF。
 * 3) プラグイン順は他のメッセージ系より下（後）を推奨。
 *
 * ■備考
 * ・選択肢/数値/アイテム選択中は自動送りしません。
 * ・外部互換パッチ（LL_CustomFade / GlobalSlowZoom）を使う場合は、本体の保護設定を参照する最新版をお使いください。
 */
(() => {
  "use strict";
  const N = "HS_FastMode";
  const P = PluginManager.parameters(N);

  const OPTION_NAME = String(P["OptionName"] || "高速モード");
  const IMPORTANT_SWITCH_ID = Number(P["ImportantSwitchId"] || 0);

  const PROTECT_OPTION_NAME = String(P["ProtectOptionName"] || "重要シーンを保護");
  const PROTECT_DEFAULT = String(P["ProtectDefault"] || "true") === "true";

  const FORCE_MESSAGE_FAST = String(P["ForceMessageFast"] || "true") === "true";
  const MSG_WAIT_DIV = Math.max(1, Number(P["MessageWaitDivisor"] || 2));
  const INT_WAIT_DIV = Math.max(1, Number(P["InterpreterWaitDivisor"] || 2));

  const AUTO_OPTION_NAME = String(P["AutoOptionName"] || "自動送り");
  const AUTO_DEFAULT = String(P["AutoEnabledByDefault"] || "false") === "true";
  const AUTO_BASE = Math.max(0, Number(P["AutoBaseWait"] || 30));
  const AUTO_PER_CHAR = Math.max(0, Number(P["AutoPerCharWait"] || 1));
  const AUTO_MIN = Math.max(0, Number(P["AutoMinWait"] || 20));
  const AUTO_MAX = Math.max(0, Number(P["AutoMaxWait"] || 240));
  const AUTO_HOTKEY_A = String(P["AutoHotkeyA"] || "true") === "true";

  // ---- 保護判定
  function importantBlocked() {
    // 保護オプションがOFFのときは、スイッチがONでもブロックしない
    if (!ConfigManager.hsProtect) return false;
    return IMPORTANT_SWITCH_ID > 0 && $gameSwitches && $gameSwitches.value(IMPORTANT_SWITCH_ID);
  }
  function fastModeEnabled() {
    return !!ConfigManager.fastMode && !importantBlocked();
  }
  function autoModeEnabled() {
    return !!ConfigManager.hsAuto && !importantBlocked();
  }

  // ---- 設定保存
  ConfigManager.fastMode = false;
  ConfigManager.hsAuto = AUTO_DEFAULT;
  ConfigManager.hsProtect = PROTECT_DEFAULT;

  const _makeData = ConfigManager.makeData;
  ConfigManager.makeData = function () {
    const c = _makeData.call(this);
    c.fastMode = this.fastMode;
    c.hsAuto = this.hsAuto;
    c.hsProtect = this.hsProtect;
    return c;
  };
  const _applyData = ConfigManager.applyData;
  ConfigManager.applyData = function (config) {
    _applyData.call(this, config);
    this.fastMode = this.readFlag(config, "fastMode");
    this.hsAuto = this.readFlag(config, "hsAuto");
    this.hsProtect = this.readFlag(config, "hsProtect");
  };

  // ---- オプション項目
  const _Window_Options_addGeneralOptions = Window_Options.prototype.addGeneralOptions;
  Window_Options.prototype.addGeneralOptions = function () {
    _Window_Options_addGeneralOptions.call(this);
    this.addCommand(OPTION_NAME, "fastMode");
    this.addCommand(AUTO_OPTION_NAME, "hsAuto");
    this.addCommand(PROTECT_OPTION_NAME, "hsProtect");
  };

  // ---- メッセージ待機短縮
  const _Window_Message_startWait = Window_Message.prototype.startWait;
  Window_Message.prototype.startWait = function (count) {
    if (fastModeEnabled() && MSG_WAIT_DIV > 1) count = Math.floor(count / MSG_WAIT_DIV);
    _Window_Message_startWait.call(this, count);
  };
  const _Window_Message_updateShowFast = Window_Message.prototype.updateShowFast;
  Window_Message.prototype.updateShowFast = function () {
    _Window_Message_updateShowFast.call(this);
    if (fastModeEnabled() && FORCE_MESSAGE_FAST) this._showFast = true;
  };

  // ---- イベント待機短縮
  const _Game_Interpreter_wait = Game_Interpreter.prototype.wait;
  Game_Interpreter.prototype.wait = function (duration) {
    if (fastModeEnabled() && INT_WAIT_DIV > 1) duration = Math.floor(duration / INT_WAIT_DIV);
    _Game_Interpreter_wait.call(this, duration);
  };

  // ---- 自動送り：ページ終了時にタイマーセット
  const _Window_Message_startPause = Window_Message.prototype.startPause;
  Window_Message.prototype.startPause = function () {
    _Window_Message_startPause.call(this);
    this._hsAutoTimer = -1;
    this._hsAutoReady = false;
    if (autoModeEnabled() && !$gameMessage.isChoice() && !$gameMessage.isNumberInput() && !$gameMessage.isItemChoice()) {
      const textLen = ($gameMessage.allText && $gameMessage.allText().length) || 0;
      let wait = AUTO_BASE + AUTO_PER_CHAR * textLen;
      wait = Math.max(AUTO_MIN, Math.min(AUTO_MAX, wait));
      if (fastModeEnabled() && MSG_WAIT_DIV > 1) wait = Math.max(AUTO_MIN, Math.floor(wait / MSG_WAIT_DIV));
      this._hsAutoTimer = wait;
    }
  };
  // ---- 更新：pause中のみカウントダウン
  const _Window_Message_update = Window_Message.prototype.update;
  Window_Message.prototype.update = function () {
    _Window_Message_update.call(this);
    if (!autoModeEnabled()) { this._hsAutoTimer = -1; this._hsAutoReady = false; return; }
    if ($gameMessage.isChoice() || $gameMessage.isNumberInput() || $gameMessage.isItemChoice()) { this._hsAutoTimer = -1; this._hsAutoReady = false; return; }
    if (Input.isTriggered("ok") || Input.isTriggered("cancel") || TouchInput.isTriggered()) { this._hsAutoTimer = -1; this._hsAutoReady = false; return; }
    if (this.pause && this._hsAutoTimer >= 0) {
      if (this._hsAutoTimer > 0) this._hsAutoTimer--;
      else { this._hsAutoReady = true; this._hsAutoTimer = -1; }
    } else { this._hsAutoTimer = -1; this._hsAutoReady = false; }
  };
  // ---- 進行判定にオートを統合
  const _Window_Message_isTriggered = Window_Message.prototype.isTriggered;
  Window_Message.prototype.isTriggered = function () {
    if (autoModeEnabled() && this.pause && this._hsAutoReady && !$gameMessage.isChoice() && !$gameMessage.isNumberInput() && !$gameMessage.isItemChoice()) {
      this._hsAutoReady = false;
      return true;
    }
    return _Window_Message_isTriggered.call(this);
  };

  // ---- Aキーで自動送りON/OFF（任意）
  if (AUTO_HOTKEY_A) {
    if (!Input.keyMapper[65]) Input.keyMapper[65] = "hs_auto_toggle"; // 'A'
    const toggleAuto = () => {
      if (SceneManager._scene && (SceneManager._scene instanceof Scene_Map || SceneManager._scene instanceof Scene_Battle)) {
        if (Input.isTriggered("hs_auto_toggle")) { ConfigManager.hsAuto = !ConfigManager.hsAuto; SoundManager.playCursor(); }
      }
    };
    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function () { _Scene_Map_update.call(this); toggleAuto(); };
    const _Scene_Battle_update = Scene_Battle.prototype.update;
    Scene_Battle.prototype.update = function () { _Scene_Battle_update.call(this); toggleAuto(); };
  }

  // ---- 互換パッチ用の公開API（任意）
  window.HS_FM = {
    msgWaitDiv: MSG_WAIT_DIV,
    intWaitDiv: INT_WAIT_DIV,
    isBlocked: () => importantBlocked(),
    fastActive: () => fastModeEnabled(),
    autoActive: () => autoModeEnabled(),
    protectEnabled: () => !!ConfigManager.hsProtect
  };
})();

